/*
 * Create simple gif files for testing
 *
 * Copyright (C) 2014 Roger Walker - smallgif.com
 * Commercial and Non-commercial is allowed, for this or it's derivatives providing you credit me
 * In the "About box" or documentation you should mention "The gif image file functions are based on work by Roger@smallgif.com"
 */
const char *version = "_$Id: gifgen.c,v 1.12 2012/02/12 03:19:32 roger Exp $_";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <assert.h>
#include "gif_ls.h"

static void create_screen_bits(gif_file_t *psInfo,unsigned uWidth,unsigned uHeight,unsigned uBits)
{	assert(uBits>=1 && uBits<=8);
	memset(psInfo,0,sizeof(*psInfo));
	psInfo->screen.width=(WORD)uWidth;
	psInfo->screen.height=(WORD)uHeight;
	psInfo->screen.bgColour=0;

	psInfo->screen.bGlobalColourTable=true;
	psInfo->screen.bitsPerPixelMinus1=(BYTE)(uBits-1);
	psInfo->screen.bitsPerColourValMinus1=7;
	psInfo->screen.sort=false;

	// Allocate global colour table
	psInfo->uColourTableBytes=sizeof(psInfo->psColourTable)*GIF_NCOLOURS(psInfo->screen);
	psInfo->psColourTable=malloc(psInfo->uColourTableBytes);
	assert(psInfo->psColourTable);
}

static void set_colour(gif_colour_entry_t *psColour,BYTE red,BYTE green,BYTE blue)
{	psColour->red=red;
	psColour->green=green;
	psColour->blue=blue;
}

// Create default screen info of given size
static void create_screen(gif_file_t *psInfo,unsigned uWidth,unsigned uHeight)
{	// Default image = 2 bits per pixel BG colour=0 colours W,R,G,B
	create_screen_bits(psInfo,uWidth,uHeight,2);

	// Palette 0 = white, 1=red, 2=green, 3=blue
	set_colour(&psInfo->psColourTable[0],0xFF,0xFF,0xFF);
	set_colour(&psInfo->psColourTable[1],0xFF,0   ,0);
	set_colour(&psInfo->psColourTable[2],0   ,0xFF,0);
	set_colour(&psInfo->psColourTable[3],0   ,0   ,0xFF);
}

// Create image of given size with local colour table of given bit depth
// bit depth of 0 = use global colour table
static gif_image_t *create_image(gif_file_t *psInfo,unsigned uWidth,unsigned uHeight,unsigned uBits)
{	gif_image_t *psImage;

	assert(uWidth<=psInfo->screen.width);
	assert(uHeight<=psInfo->screen.height);
	assert(uBits<=8);

	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=0;
	psImage->image.top=0;
	psImage->image.width=(WORD)uWidth;
	psImage->image.height=(WORD)uHeight;
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);

	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=1;	// Leave image
	psImage->control.delaytime=500;	// 2.5 seconds

	if (uBits>0)
	{	psImage->image.bLocalColourTable=true;
		psImage->image.bitsPerPixelMinus1=(BYTE)(uBits-1);
		psImage->uColourTableBytes=GIF_NCOLOURS(psImage->image)*sizeof(*psImage->psColourTable);
		psImage->psColourTable=malloc(psImage->uColourTableBytes);
		assert(psImage->psColourTable);
		memset(psImage->psColourTable,0,psImage->uColourTableBytes);	// Default = All black
	} else
	{	psImage->image.bLocalColourTable=false;
		psImage->image.bitsPerPixelMinus1=0;
		psImage->uColourTableBytes=0;
		psImage->psColourTable=NULL;
	}
	return psImage;
}

static gif_image_t *create_image_transcol0(gif_file_t *psInfo,unsigned uWidth,unsigned uHeight,unsigned uBits)
{	gif_image_t *psImage=create_image(psInfo,uWidth,uHeight,uBits);
	if (psImage)
	{	psImage->control.bTransparent=true;
		psImage->control.byTransparentColour=0;
	}
	return psImage;
}

static void create_colours_frombase(gif_image_t *psImage,BYTE byRed,BYTE byGreen,BYTE byBlue,BYTE byRedInc,BYTE byGreenInc,BYTE byBlueInc,unsigned uNColours)
{	gif_colour_entry_t *psColourTable=psImage->psColourTable;
	gif_colour_entry_t sColour;
	assert(psImage->image.bLocalColourTable);
	assert(uNColours>=1);
	assert(uNColours<=GIF_NCOLOURS(psImage->image));
	assert(psColourTable);
	assert((byRed+(byRedInc*(uNColours-1)))<256);
	assert((byGreen+(byGreenInc*(uNColours-1)))<256);
	assert((byBlue+(byBlueInc*(uNColours-1)))<256);
	assert(psColourTable);

	sColour.red=byRed;
	sColour.green=byGreen;
	sColour.blue=byBlue;
	for ( ; uNColours>0 ; uNColours--,psColourTable++)
	{	sColour.red+=byRedInc;
		sColour.green+=byGreenInc;
		sColour.blue+=byBlueInc;
		*psColourTable=sColour;
	}
}

static void create_colours(gif_image_t *psImage,bool bRedInc,bool bGreenInc,bool bBlueInc,unsigned uNColours)
{	create_colours_frombase(psImage,0,0,0,bRedInc ? 1 : 0,bGreenInc ? 1 : 0,bBlueInc ? 1 : 0,uNColours);
}

static void fill_area_sequence(gif_image_t *psImage,unsigned uLeft,unsigned uTop,unsigned uWidth,unsigned uHeight,BYTE byStartColour)
{	unsigned uXPos,uYPos;
	assert((uTop+uHeight)<=psImage->image.height);
	assert((uLeft+uWidth)<=psImage->image.width);
	for (uYPos=0 ; uYPos<uHeight ; uYPos++)
	for (uXPos=0 ; uXPos<uWidth ; uXPos++)
	{	// Set pixel
		psImage->pLines[uYPos+uTop][uXPos+uLeft]=byStartColour++;
		if (psImage->image.bLocalColourTable && psImage->image.bitsPerPixelMinus1!=7 && byStartColour==GIF_NCOLOURS(psImage->image))
		{	byStartColour=0;
		}
	}
}

// Don't repeat any pairs of bytes
static void fill_area_non_compress(gif_image_t *psImage,unsigned uLeft,unsigned uTop,unsigned uWidth,unsigned uHeight,BYTE byStartColour,BYTE byNColours)
{	unsigned uXPos,uYPos;
	bool aabSequence[256][256];
	BYTE byColour;
	assert((uTop+uHeight)<=psImage->image.height);
	assert((uLeft+uWidth)<=psImage->image.width);
	memset(aabSequence,0,sizeof(aabSequence));

	byColour=0;
	for (uYPos=0 ; uYPos<uHeight ; uYPos++)
	for (uXPos=0 ; uXPos<uWidth ; uXPos++)
	{	// Set pixel
		if (uYPos || uXPos)
		{	// If not first pixel, do it
			BYTE byNext;
			for (byNext=1 ; byNext<=byNColours ; byNext++)
				if (!aabSequence[byColour][(byColour+byNext)%byNColours]) break;
			byNext=(BYTE)((byColour+byNext)%byNColours);
			aabSequence[byColour][byNext]=true;
			byColour=byNext;
		}
		assert(byColour<byNColours);
		psImage->pLines[uYPos+uTop][uXPos+uLeft]=(BYTE)(byStartColour+byColour);
	}
}

// Write a simple to compress byte sequence to the destination
static void fill_area_compressable(BYTE *pbyDest,BYTE byStartVal,BYTE byEndVal,unsigned uLen)
{	unsigned uPos=0;
	unsigned uSequenceLen=2;
	while (uPos<uLen)
	{	unsigned uLoop;
		for (uLoop=0 ; uPos<uLen && uLoop<uSequenceLen && (byStartVal+uLoop)<=byEndVal ; uLoop++)
		{	pbyDest[uPos++]=(BYTE)(byStartVal+uLoop);
		}
		uSequenceLen++;
	}
}

// Create simple stripe, if byNbpp==0 then it's global palette
static gif_image_t *add_simple_image(gif_file_t *psInfo,unsigned uSize,BYTE byNbpp,BYTE byBGCol,BYTE byTLCol,BYTE byTRCol)
{	gif_image_t *psImage=create_image(psInfo,uSize,uSize,byNbpp);

	if (psImage)
	{	unsigned uPos;
		psImage->control.bTransparent=false;
		psImage->control.byTransparentColour=0xFF;

		if (byBGCol!=0)
			memset(psImage->pLines[0],byBGCol,psImage->uNPixels);

		if (byTLCol!=byBGCol)
		{	for (uPos=0 ; uPos<uSize ; uPos++)
			{	psImage->pLines[uPos][uPos]=byTLCol;
			}
		}

		if (byTRCol!=byBGCol)
		{	for (uPos=0 ; uPos<uSize ; uPos++)
			{	psImage->pLines[uPos][(uSize-1)-uPos]=byTRCol;
			}
		}
	}
	return psImage;
}

// Simple X image
static void file0(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,100,100);
	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=0;
	psImage->image.top=0;
	psImage->image.width=psInfo->screen.width;
	psImage->image.height=psInfo->screen.height;
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
}


// Simple 2 frame animated image
static void file1(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,100,100);
	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=0;
	psImage->image.top=0;
	psImage->image.width=psInfo->screen.width;
	psImage->image.height=psInfo->screen.height;
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=2;
	psImage->control.delaytime=500;

	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=(WORD)(psInfo->screen.width/3);
	psImage->image.top=0;
	psImage->image.width=(WORD)(psInfo->screen.width/3);
	psImage->image.height=(WORD)(psInfo->screen.height/3);
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=2;
	psImage->control.delaytime=500;
}


// Simple 3 frame animated image
static void file2(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,100,100);
	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=0;
	psImage->image.top=0;
	psImage->image.width=psInfo->screen.width;
	psImage->image.height=psInfo->screen.height;
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=2;
	psImage->control.delaytime=500;

	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=(WORD)(psInfo->screen.width/3);
	psImage->image.top=0;
	psImage->image.width=(WORD)(psInfo->screen.width/3);
	psImage->image.height=(WORD)(psInfo->screen.height/3);
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=2;
	psImage->control.delaytime=500;

	psImage=gif_add_image_space(psInfo);
	assert(psImage);
	psImage->image.left=0;
	psImage->image.top=(WORD)(psInfo->screen.height/3);
	psImage->image.width=(WORD)(psInfo->screen.width/3);
	psImage->image.height=(WORD)(psInfo->screen.height/3);
	if (gif_allocate_image_memory(psImage)) assert(0);
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<psImage->image.width ; uPos++)
	{	psImage->pLines[uPos][uPos]=1;
		psImage->pLines[uPos][(psImage->image.width-1)-uPos]=2;
	}
	psImage->bHaveControl=true;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=99;
	psImage->control.disposal=2;
	psImage->control.delaytime=500;
}


// Create image with many colours
static void file3(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,32,32);

	// Set top left = Shades of red
	psImage=create_image(psInfo,32,32,8);
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,16,16,0);

	// Create top right = Shades of green
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,16,0,16,16,0);
}

// Many colours requiring mode 3
static void file4(gif_file_t *psInfo)
{	gif_image_t *psImage;

	// Top left = red, top right = green
	file3(psInfo);

	// Middle = blue
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,8,8,16,16,0);

	// Add just bottom right = blue
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,16,16,16,16,0);
}

// Many colours requiring multiple mode 3
static void file5(gif_file_t *psInfo)
{	gif_image_t *psImage;

	file3(psInfo);

	// Add just bottom right = blue
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,16,8,16,16,0);

	// middle , middle = cyan
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=3;
	create_colours(psImage,true,false,true,256);
	fill_area_sequence(psImage,8,8,16,16,0);

	// middle , middle = yellow
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=3;
	create_colours(psImage,true,true,false,256);
	fill_area_sequence(psImage,8,8,16,16,0);

	// bottom left = blue
	psImage=create_image(psInfo,32,32,8);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,8,16,16,0);
}

// Easy transcol test
static void file6(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,32,24);

	// Image 0 = All reds
	psImage=create_image(psInfo,32,24,8);
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,32,24,0);
	psImage->control.disposal=1;			// Leave

	// Image 1 = mid left image with border = greens
	psImage=create_image(psInfo,18,18,8);
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=3;
	psImage->image.top=3;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=2;		// Clear area to background, leaving hole

	// Image 2 = mid bottom = blue
	psImage=create_image(psInfo,16,16,8);
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=8;
	psImage->image.top=8;
	psImage->control.disposal=1;		// Leave
}

// Not compressable, no byte pairs are repeated, no colours in second image match first
static void file7(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPix;

	create_screen(psInfo,8,8);
	psInfo->screen.bitsPerPixelMinus1=4-1;

	// Re-allocate global colour table
	psInfo->uColourTableBytes=sizeof(psInfo->psColourTable)*GIF_NCOLOURS(psInfo->screen);
	psInfo->psColourTable=realloc(psInfo->psColourTable,psInfo->uColourTableBytes);
	assert(psInfo->psColourTable);

	// Create global palette, 16 colours where bit 0=red, 1=green, 2=blue, then grey versions
	for (uPix=0 ; uPix<8 ; uPix++)
	{	psInfo->psColourTable[uPix].red=(uPix&1) ? 0xFF : 0;
		psInfo->psColourTable[uPix].green=(uPix&2) ? 0xFF : 0;
		psInfo->psColourTable[uPix].blue=(uPix&4) ? 0xFF : 0;
	}
	for (uPix=0 ; uPix<8 ; uPix++)
	{	psInfo->psColourTable[8+uPix].red=(uPix&1) ? 0xC0 : 0x40;
		psInfo->psColourTable[8+uPix].green=(uPix&2) ? 0xC0 : 0x40;
		psInfo->psColourTable[8+uPix].blue=(uPix&4) ? 0xC0 : 0x40;
	}

	// Create first image
	psImage=create_image(psInfo,psInfo->screen.width,psInfo->screen.height,0);
	fill_area_non_compress(psImage,0,0,psImage->image.width,psImage->image.height,0,8);

	// Create second image
	psImage=create_image(psInfo,psInfo->screen.width,psInfo->screen.height,0);
	fill_area_non_compress(psImage,0,0,psImage->image.width,psImage->image.height,4,8);
}

// Multi mode 3 test with restore frames
static void file8(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,32,24);

	// Create
	// R. RG RG RG RG
	// .. .. B. bb .B // overlaps 50% with above in Y, bb=mid

	// Image 0 = top left = reds
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=1;
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;

	// Image 1 = Create top right = greens
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=1;
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,16,0,16,16,0);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;

	// Image 2 = top right = blue
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,16,0,16,16,0);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	// Restores image afterwards to red + green

	for (uPos=0 ; uPos<=16 ; uPos+=8)
	{	// Image 3..4..5 = left,mid,right overlapped with top row = blue
		psImage=create_image(psInfo,32,24,8);
		psImage->control.disposal=3;
		create_colours(psImage,false,false,true,256);
		fill_area_sequence(psImage,(16-uPos),8,16,16,0);
		psImage->control.bTransparent=true;
		psImage->control.byTransparentColour=0;
	}
}

// Should be same result as 8, but with cropped source images
static void file9(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,32,24);

	// Create
	// R. RG RG RG RG
	// .. .. B. bb .B // overlaps 50% with above in Y, bb=mid

	// Image 0 = top left = reds
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=1;
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;

	// Image 1 = Create top right = greens
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=1;
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=16;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;

	// Image 2 = top right = blue
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=16;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	// Restores image afterwards to red + green

	for (uPos=0 ; uPos<=16 ; uPos+=8)
	{	// Image 3..4..5 = left,mid,right overlapped with top row = blue
		psImage=create_image(psInfo,16,16,8);
		psImage->control.disposal=3;
		create_colours(psImage,false,false,true,256);
		fill_area_sequence(psImage,0,0,16,16,0);
		psImage->image.left=(WORD)(16-uPos);
		psImage->image.top=8;
		psImage->control.bTransparent=true;
		psImage->control.byTransparentColour=0;
	}
}

// Hard mode test for 1,2,3
static void file10(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,32,24);

	// Image 0 = All reds
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=1;
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,32,24,0);

	// Image 1 = mid left image with border = greens
	psImage=create_image(psInfo,18,18,8);
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=3;
	psImage->image.top=3;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=2;		// Clear area to background, leaving hole

	// Image 2 = mid bottom = blue
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=8;
	psImage->image.top=8;

	// Image 3 = right bottom = magenta
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,true,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=16;
	psImage->image.top=8;
}

// Multi mode 3 test
static void file11(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,32,24);

	// Image 0 = All reds
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=1;
	create_colours(psImage,true,false,false,256);
	fill_area_sequence(psImage,0,0,32,24,0);

	// Image 1 = left greens
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,true,false,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.top=4;

	// Image 2 = left blue (on same area as green)
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.top=4;

	// Image 3 = right blue (so red restored on left)
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours(psImage,false,false,true,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=16;
	psImage->image.top=4;
}

// Simple optional transcol test
static void file12(gif_file_t *psInfo)
{	gif_image_t *psImage;
	unsigned uPos;

	create_screen(psInfo,16,2);
	// Global colours, 0=white, 1=red, 2=green, 3=blue

	// Image 0 = Mainly white with 0,1,2,3 sequences in middle
	psImage=create_image(psInfo,16,2,2);
	psImage->image.bLocalColourTable=false;
	psImage->control.bTransparent=false;
	psImage->control.byTransparentColour=0xFF;
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<12 ; uPos++)
	{	psImage->pLines[0][uPos+2]=psImage->pLines[1][uPos+2]=(BYTE)(uPos&3);
	}

	// Image 1 = Other than corners, can all be transcol
	psImage=create_image(psInfo,16,2,2);
	psImage->image.bLocalColourTable=false;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<12 ; uPos++)
	{	psImage->pLines[0][uPos+2]=psImage->pLines[1][uPos+2]=(BYTE)(uPos&3);
	}
	psImage->pLines[0][15]=1;
	psImage->pLines[1][0]=2;
	psImage->pLines[1][15]=3;

	// Image 2 = Other than corners, alternate pixels can be transcol
	psImage=create_image(psInfo,16,2,2);
	psImage->image.bLocalColourTable=false;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	memset(psImage->pLines[0],0,psImage->uNPixels);
	for (uPos=0 ; uPos<14 ; uPos++)
	{	psImage->pLines[0][uPos+1]=psImage->pLines[1][uPos+1]=(BYTE)(uPos&3);
	}
	psImage->pLines[0][15]=2;
	psImage->pLines[1][0]=3;
	psImage->pLines[1][15]=1;
}

// Hard mode test for 1,2,3
static void file13(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,32,24);

	// Image 0 = Cover image with 256 different reds
	psImage=create_image(psInfo,32,24,8);
	psImage->control.disposal=1;
	create_colours_frombase(psImage,0,0x10,0x10, 1,0,0,256);
	fill_area_sequence(psImage,0,0,32,24,0);

	// Image 1 = mid left image with border = greens
	// Previous has to be mode 1, as we don't have enough colours to draw the reds
	psImage=create_image(psInfo,18,18,8);
	create_colours_frombase(psImage,0x20,0,0x20, 0,1,0,256);
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=3;
	psImage->image.top=3;
	psImage->control.bTransparent=true;
	psImage->control.byTransparentColour=0;
	psImage->control.disposal=2;		// Clear area to background, leaving hole

	// Image 2 = mid left = blue (20x20 so that this can't be mode 2)
	// Previous has to be mode 2, as next image will use the 16x16 cleared area
	psImage=create_image(psInfo,20,20,8);
	psImage->control.disposal=3;
	create_colours_frombase(psImage,0x30,0x30,0, 0,0,1,256);
	fill_area_sequence(psImage,0,0,20,20,0);
	psImage->image.left=2;
	psImage->image.top=2;
	// This will restore, leaving the transparent hole from the 16x16 image 1 (mode 2)

	// Image 3 = right bottom = magenta
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours_frombase(psImage,0,0x40,0x40, 1,0,0,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=16;
	psImage->image.top=8;

	// Image 4 = mid mid = magenta
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;
	create_colours_frombase(psImage,0x50,0,0x50, 0,1,0,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=12;
	psImage->image.top=4;

	// Image 5 = right mid = magenta
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=2;
	create_colours_frombase(psImage,0x60,0x60,0, 0,0,1,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=12;
	psImage->image.top=8;


	// Image 6 = 20x20 image with colour 0 border
	psImage=create_image(psInfo,20,20,8);
	psImage->control.disposal=1;
	create_colours_frombase(psImage,0,0x70,0x70, 1,0,0,256);
	fill_area_sequence(psImage,2,2,16,16,0);	// Border already set to 0
	psImage->image.left=4;
	psImage->image.top=4;

	// Image 7 = 18x18 image with previous colour 0 border
	psImage=create_image(psInfo,18,18,8);
	psImage->control.disposal=2;		// Delete afterwards
	create_colours_frombase(psImage,0x80,0,0x80, 0,1,0,256);
	psImage->psColourTable[0]=psImage[-1].psColourTable[0];	// Copy previous colour 0
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=5;
	psImage->image.top=5;

	// Image 8 = 18x18 image with previous colour 0 border
	psImage=create_image(psInfo,18,18,8);
	psImage->control.disposal=3;		// Restore previous (including background)
	create_colours_frombase(psImage,0x90,0x90,0, 0,0,1,256);
	psImage->psColourTable[0]=psImage[-1].psColourTable[0];	// Copy previous colour 0
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=5;
	psImage->image.top=5;

	// Image 8 = 18x18 image with previous colour 0 border
	psImage=create_image(psInfo,18,18,8);
	psImage->control.disposal=1;
	create_colours_frombase(psImage,0,0xA0,0xA0, 1,0,0,256);
	psImage->psColourTable[0]=psImage[-1].psColourTable[0];	// Copy previous colour 0
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=10;
	psImage->image.top=5;
}

// Hard 1,2,3 test with less images
static void file14(gif_file_t *psInfo)
{	gif_image_t *psImage;

	create_screen(psInfo,24,24);

	// Image 0 = Cover image with 256 different reds
	psImage=create_image(psInfo,24,24,8);
	psImage->control.disposal=1;
	create_colours_frombase(psImage,0,0x10,0x10, 1,0,0,256);
	fill_area_sequence(psImage,0,0,24,24,0);

	// Image 1 = top green with border
	psImage=create_image(psInfo,18,18,8);
	create_colours_frombase(psImage,0x20,0,0x20, 0,1,0,256);
	fill_area_sequence(psImage,1,1,16,16,0);	// Border already set to 0
	psImage->image.left=3;
	psImage->image.top=3;
	psImage->control.bTransparent=false;
	psImage->control.disposal=2;		// Clear area to background, leaving hole

	// Image 2 = mid green with border left right
	// Previous has to be mode 2, as next image will use the 18x18 cleared area
	psImage=create_image(psInfo,18,18,8);
	psImage->control.disposal=3;
	create_colours_frombase(psImage,0x20,0,0x20, 0,1,0,256);
	fill_area_sequence(psImage,1,0,16,18,0);
	psImage->image.left=3;
	psImage->image.top=4;
	// This will restore, leaving the transparent line from the 16x16 image 1 (mode 2)

	// Image 3 = bottom = blue
	psImage=create_image(psInfo,16,16,8);
	psImage->control.disposal=3;	// Doesn't actually care
	create_colours_frombase(psImage,0,0x30,0x30, 1,0,0,256);
	fill_area_sequence(psImage,0,0,16,16,0);
	psImage->image.left=4;
	psImage->image.top=8;

	// Double check 1 and 2 use the same colours, then put them in global
	assert(memcmp(psInfo->psImages[1].psColourTable,psInfo->psImages[2].psColourTable,256*sizeof(*psInfo->psImages[1].psColourTable))==0);
	assert(psInfo->psImages[1].image.bitsPerPixelMinus1==7);
	assert(psInfo->psImages[2].image.bitsPerPixelMinus1==7);
	if (psInfo->psColourTable)
		free(psInfo->psColourTable);
	psInfo->psColourTable=psInfo->psImages[1].psColourTable;
	psInfo->screen.bitsPerPixelMinus1=psInfo->psImages[1].image.bitsPerPixelMinus1;

	psInfo->psImages[1].image.bLocalColourTable=false;
	psInfo->psImages[1].psColourTable=NULL;

	psInfo->psImages[2].image.bLocalColourTable=false;
	free(psInfo->psImages[2].psColourTable);
	psInfo->psImages[2].psColourTable=NULL;
}

int main(int argc,char *argv[])
{	gif_file_t sGifImage;
	unsigned uMode;
	unsigned uSize=0;
	struct
	{	void (* pfnCreatImage)(gif_file_t *psInfo);
		char const *pszName;
	} asTests[]=
	{	{	file0,	"Simple X image" },
		{	file1,	"Simple 2 frame animated image" },
		{	file2,	"Simple 3 frame animated image" },
		{	file3,	"Create image with many colours" },
		{	file4,	"Many colours requiring mode 3" },
		{	file5,	"Many colours requiring multiple mode 3" },
		{	file6,	"Hard transcol test" },
		{	file7,	"Not compressable, no repeated colour pairs, no matches from frame 1 to 2" },
		{	file8,	"Multi mode 3 test with restore frames" },
		{	file9,	"Multi mode 3 test with restore frames, cropped" },
		{	file10,	"Forced mode test for 1,2,3" },
		{	file11,	"Forced multi mode 3" },
		{	file12,	"Simple optional transcol test" },
		{	file13,	"Forced mode test for 1,2,3 (complex)" },
		{	file14,	"Forced modes 1,2,3 (4 image)" }
	};

	if (argc>=3 && argc<=4)
	{	uMode=atoi(argv[1]);
		if (argc>=4)
		{	uSize=atoi(argv[3]);
		}
		if (argc>3 && (uMode!=13 || uSize==0))
			uMode=99;
	} else
	{	uMode=99;
	}
	if (uMode>=NENTRIES(asTests))
	{	fprintf(stderr,"SYNTAX: gifgen <image #> <filename> [<size>]\n");
		for (uMode=0 ; uMode<NENTRIES(asTests) ; uMode++)
			printf("%4u: %s\n",uMode,asTests[uMode].pszName);
		return 99;
	}

	printf("Writing '%s' as image %u (%s)\n",argv[2],uMode,asTests[uMode].pszName);
	memset(&sGifImage,0,sizeof(sGifImage));
	if (uSize)
		sGifImage.screen.width=sGifImage.screen.height=(WORD)uSize;
	(asTests[uMode].pfnCreatImage)(&sGifImage);
	{	// If global not used, move image 0 local into global
		unsigned uImage;
		for (uImage=0 ; uImage<sGifImage.uNImages ; uImage++)
			if (!sGifImage.psImages[uImage].image.bLocalColourTable)
				break;
		if (uImage>sGifImage.uNImages)
		{	assert(sGifImage.psColourTable);
			assert(sGifImage.screen.bGlobalColourTable);
			assert(sGifImage.psImages[0].psColourTable);
			assert(sGifImage.psImages[0].image.bLocalColourTable);
			free(sGifImage.psColourTable);
			sGifImage.psColourTable=sGifImage.psImages[0].psColourTable;
			sGifImage.psImages[0].psColourTable=NULL;
			sGifImage.psImages[0].image.bLocalColourTable=false;
			sGifImage.screen.bitsPerPixelMinus1=sGifImage.psImages[0].image.bitsPerPixelMinus1;
		}
	}
	{	gif_options_t sOptions;
		memset(&sOptions,0,sizeof(sOptions));
		sOptions.byCompressionLevel=1;
		gif_write_file(&sGifImage,&sOptions,argv[2]);
	}
	gif_free(&sGifImage);
	return 0;
}
