#ifndef DONE_GIF_LOAD_H	// [
#define DONE_GIF_LOAD_H	"_$Id: gif_load.h,v 1.49 2014/01/16 13:11:23 roger Exp $_"

/*
 * GIF load functions
 *
 * Copyright (C) 2014 Roger Walker - smallgif.com
 * Commercial and Non-commercial is allowed, for this or it's derivatives providing you credit me
 * In the "About box" or documentation you should mention "The gif image file functions are based on work by Roger@smallgif.com"
*/
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include "gif_type.h"

#ifdef __cplusplus
extern "C" {
#endif

/*************************** defines / types **************************/

#ifdef PACK_USEPRAGMA
#pragma pack(1)
// Don't warn about non-int bit packed structures
#pragma warning(push,1)
#endif

// Must be at start of stream
typedef struct gif_start_tag
{	char sig_G;
	char sig_I;
	char sig_F;
	char ver[3];	// Must be 87a or 89a
} PACK_ATTRIB gif_start_t;

// Must be after gif_start_t
typedef struct gif_logical_screen_tag
{	WORD width;
	WORD height;
	// LSB first
	BYTE	bitsPerPixelMinus1:3,		// 2^(bitsPerPixelMinus1+1) = #colours in pallette
			sort:1,
			bitsPerColourValMinus1:3,	// 2^(bitsPerColourVal+1) = bits per colour channel
			bGlobalColourTable:1;
	BYTE bgColour;
	BYTE aspect;	// Aspect Ratio = (Pixel Aspect Ratio + 15) / 64 (0==not known)
} PACK_ATTRIB gif_logical_screen_t;

// If bGlobalColourTable!=0, must follow gif_logical_screen_t
// Will have 2^(bitsPerPixelMinus1+1) entries of gif_colour_entry_t (each 3 bytes)
typedef struct gif_colour_entry_tag
{	BYTE red;
	BYTE green;
	BYTE blue;
} PACK_ATTRIB gif_colour_entry_t;

// 0x2C -> Image, followed by gif_image_descriptor_t, then local colour table (if bit set), then code size, then image data
typedef struct gif_image_descriptor_tag
{	WORD left;
	WORD top;
	WORD width;
	WORD height;
	// LSB first
	BYTE	bitsPerPixelMinus1:3,	// 2^(bitsPerPixelMinus1+1) = #colours in pallette
			Reserved1:2,
			bSorted:1,
			bInterlaced:1,
			bLocalColourTable:1;
} PACK_ATTRIB gif_image_descriptor_t;

// If bLocalColourTable!=0, must follow gif_image_descriptor_t
// Will have 2^(bitsPerPixelMinus1+1) entries of gif_colour_entry_t (each 3 bytes)

// 0x3B = End of images


// GIF89a Extensions are of the form 0x21 <block type>
// followed by a number of blocks of the form <len> <len bytes of data>
// terminated with a 0x00 length

// Extension F9 = control block
typedef struct gif_ext_graphics_control_label_tag
{	// LSB first
	BYTE	bTransparent:1,
			bUserinput:1,
			disposal:3,		// After display 0-Do nothing, 1-Leave image, 2-set bgColour, 3-Restore previous image
			Reserved:3;
	WORD delaytime;			// 100ths of a sec
	BYTE byTransparentColour;
} PACK_ATTRIB gif_ext_graphics_control_label_t;

// Extension FE = comment block

// Extension 01 = plain text data, first block = header, rest = text
typedef struct gif_ext_plain_text_tag
{	WORD left;
	WORD top;
	WORD width;
	WORD height;
	BYTE charwidth;
	BYTE charheight;
	BYTE fgcolour;
	BYTE bgcolour;
} PACK_ATTRIB gif_ext_plain_text_t;

// Extension FF = application data, first block=header, rest = data blocks
typedef struct gif_ext_application_tag
{	BYTE application[8];
	BYTE applicationcode[3];
} PACK_ATTRIB gif_ext_application_t;

#ifdef PACK_USEPRAGMA
#pragma warning(pop)
#pragma pack()
#endif

typedef struct gif_image_tag
{	gif_image_descriptor_t image;
	gif_colour_entry_t *psColourTable;
	unsigned uColourTableBytes;

	unsigned uNPixels;
	BYTE **pLines;

	bool bHaveControl;
	gif_ext_graphics_control_label_t control;

	bool bHaveNetscapeLoop;
	WORD wNumLoops;
} gif_image_t;

#ifndef gif_options_t
typedef struct
{	BYTE byCompressionLevel;
	bool bMissingInitialResets;
	bool bNoOverrun;
	bool bNoMode3;
} gif_options_t;
#endif

typedef struct gif_file_tag
{	gif_logical_screen_t screen;
	gif_colour_entry_t *psColourTable;
	unsigned uColourTableBytes;
	unsigned uNImages;
	gif_image_t *psImages;

	// Found during file load
	unsigned uNWarnings;
	gif_options_t sOptions;

	// Used during load/save operations
	void *pPrivate;
} gif_file_t;


/********************** Public function definitions *************************/

extern bool gif_allocate_image_memory(gif_image_t *psImage);
extern gif_image_t *gif_add_image_space(gif_file_t *psFile);
extern void gif_free_image(gif_image_t *psImage);

extern bool gif_read_images(gif_file_t *psGifInfo,
					bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle),
					void	*pDataHandle);
extern bool gif_read_images_nowarns(gif_file_t *psGifInfo,
					bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle),
					void	*pDataHandle);
extern bool gif_read_data_from_file(BYTE *pDest,unsigned uNBytes,void *pDataHandle);
extern bool gif_read_file(gif_file_t *pDest,const char *psFileName);
extern void gif_free(gif_file_t *psFile);

extern bool gif_is_gif89(gif_file_t const *psGifInfo);

/************************** Public Macros *******************************/

#define GIF_NCOLOURS(a) (2U<<((a).bitsPerPixelMinus1))
#define GIF_CMP_COLOUR(a,b) ((a).red==(b).red && (a).green==(b).green && (a).blue==(b).blue)
#define GIF_CMP_FULLSCREEN(s,i) ((i).left==0 && (i).top==0 && (i).width==(s).width && (i).height==(s).height)
// 0 - No disposal specified. The decoder is not required to take any action.
// 1 - Do not dispose. The graphic is to be left in place.
// 2 - Restore to background color. The area used by the graphic must be restored to the background color.
// 3 - Restore to previous.
#define GIF_LOOKUP_DISPOSAL(m) ((m)==0 ? "none" : (m)==1 ? "leave" : (m)==2 ? "restore" : (m)==3 ? "previous" : "Unknown")

/*************************** Private function definitions **************************/
#ifdef GIF_ASSERT
#undef GIF_ASSERT
#endif
#if defined(GIF_LOAD_DEBUG) || defined(_DEBUG)
#define GIF_ASSERT(a) assert(a)
#else
#define GIF_ASSERT(a)
#endif

#if (defined(_DEBUG) && !defined(GIF_LOAD_STATS))
static unsigned const guGifLoadStatLevel=1;
#define GIF_LOAD_STATS guGifLoadStatLevel
#endif

#ifndef NO_FUNCTIONS	// [

#define GIF_PRIVATE_BAD_COLOUR	0x1000

static void gif_private_printf(char const *pFormat,...)
{	static unsigned uLastLen=0;
	va_list psArg;
	unsigned uNChars;
	unsigned uBack;
	va_start( psArg, pFormat);
	uNChars=vfprintf(stderr,pFormat,psArg);
	va_end(psArg);
	for (uBack=uNChars ; uBack<uLastLen ; uBack++) fputc(' ',stderr);
	for (uBack=max(uLastLen,uNChars) ; uBack>0 ; uBack--) fputc('\b',stderr);
	uLastLen=uNChars;
}

/* Read down each column, from left to right
 *	0
 *				1
 *			2
 *				3
 *		4
 *				5
 *			6
 *				7
 *	8
 *				9
 *			10
 *				11
 *		12
 *				13
 *			14
 *				15
 *	16
 *				17
 *			18
 *				19
 *		20
 *				21
 *			22
 *				23
 *	24
 * Only need when loading and saving, the ppLines[] entries are in image display order
 */
static unsigned gif_private_lookup_interlaced_line(gif_image_descriptor_t const *psImage,unsigned uLine)
{	GIF_ASSERT(psImage);
	GIF_ASSERT(uLine<psImage->height);
	if (!psImage->bInterlaced)
	{	return uLine;
	} else
	{	unsigned uYPos=uLine*8;
		if (uYPos>=psImage->height)
		{	uLine-=(psImage->height+7)/8;
			uYPos=uLine*8+4;
			if (uYPos>=psImage->height)
			{	uLine-=(psImage->height+7-4)/8;
				uYPos=uLine*4+2;
				if (uYPos>=psImage->height)
				{	uLine-=(psImage->height+3-2)/4;
					uYPos=uLine*2+1;
				}
			}
		}
		GIF_ASSERT(uYPos<psImage->height);
		return uYPos;
	}
}

#if defined(GIF_LOAD_DEBUG) || defined(GIF_DEBUG) || defined(GIF_LS_DEBUG)
// Return true if OK
static bool gif_private_interlaced_correct(gif_image_t const *psImage)
{
	// If only 2 lines, always correct
	if (psImage->image.height<=2) return true;

	if (!psImage->image.bInterlaced)
	{	// Line 1 will be after line 0
		//return SET_BOOL(psImage->pLines[1]==(psImage->pLines[0]+psImage->image.width));
		if (psImage->pLines[1]==(psImage->pLines[0]+psImage->image.width)) return true;
	} else
	{	if (psImage->pLines[1]!=(psImage->pLines[0]+psImage->image.width)) return true;
	}

	{	unsigned uLine;
		for (uLine=0 ; uLine<psImage->image.height ; uLine++)
		{	unsigned uIntLine=gif_private_lookup_interlaced_line(&psImage->image,uLine);
			printf("%u = %u -> 0x%p 0x%p\n",uLine,uIntLine,psImage->pLines[uLine],psImage->pLines[uIntLine]);
		}
	}
	return false;
}
#endif

/*
 * Gif Reading structures/functions
 */
#define GIF_LZ_BITS			12
#define GIF_LZ_NCODES		(1<<GIF_LZ_BITS)
#define GIF_NO_SUCH_CODE	(0x7FFF)	/* Impossible code, to signal empty. */
#define GIF_NETSCAPE_ID		"NETSCAPE2.0"

typedef unsigned short gif_code_t;

typedef struct gif_private_read_data_tag
{	gif_code_t	codePrevious;		/* The code before the current code. */
	gif_code_t	codeNext;			/* The next code algorithm can generate. */

	unsigned uNPixvals;				/* 0..uNPixvals-1 = Pixels, uNPixvals+0=CLEAR, uNPixvals+1=EOI */

	BYTE	byInitialBitsPerCode;	/* Initially = bits per pixel */
	BYTE	byBitsPerCode;			/* Bits per code */
	BYTE	byCurrentBits;			/* The number of bits required to represent the current code. */

	// 32-bit buffer of read bytes
	BYTE	nDataBits;				/* Number of bits valid in dwDataBits. */
	DWORD	dwDataBits;				/* For bytes decomposition into codes (worst case = 14bit read with 5 in buffer, needs 21 bits). */

	// Read from file, Buf[0]= number of bytes in buffer
	BYTE	abyBuf[256];				/* Compressed data is buffered here. */

	// Buffer of stacked bytes
	unsigned uNDecoded;				/* StackPtr = When we decode a block, the additional bytes are stored here */
	BYTE	abyDecoded[GIF_LZ_NCODES];	/* abyDecoded pixels are stacked here. */

	// Linked list to chain bytes
	WORD	awPrefix[GIF_LZ_NCODES+1];
	BYTE	abySuffix[GIF_LZ_NCODES+1];	/* So we can trace the codes. */

#ifdef GIF_LOAD_STATS
	unsigned uReadBits;
	unsigned uReadCodes;
	unsigned uWrittenPixels;

	unsigned uLastDispReadBits;
	unsigned uLastDispReadCodes;
	unsigned uLastDispWrittenPixels;
#endif

	// Read functions
	void	*pDataHandle;
	bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle);	// Returns true on error

	bool	bQuiet;
	unsigned uNWarnings;

	// Track how many are read
	unsigned uReadBytes;
} gif_private_read_data_t;

// Wrapped file read function, which keeps track of bytes read
static bool gif_private_data_read(gif_private_read_data_t *psDecomp,void *pDest,unsigned uNBytes,void *pDataHandle)
{	if (psDecomp->read_data((BYTE *)pDest,uNBytes,pDataHandle))
		return true;
	psDecomp->uReadBytes+=uNBytes;
	return false;
}

// Returns true on error
static bool gif_private_read_colour_table(gif_colour_entry_t **ppsColours,unsigned *puColourTableBytes,BYTE TableBytes,gif_private_read_data_t *psDecomp)
{	const unsigned uEntries=1<<(TableBytes+1);
	const unsigned uNBytes=uEntries*sizeof(**ppsColours);
	if ((*ppsColours=malloc(uNBytes))==NULL)
	{	fprintf(stderr,"Error: Out of memory\n");
		return true;
	}
	*puColourTableBytes=uNBytes;
	if (gif_private_data_read(psDecomp,(BYTE*)*ppsColours,uNBytes,psDecomp->pDataHandle))
	{	free(*ppsColours);
		*ppsColours=NULL;
		return true;
	}
	return false;
}

static void gif_private_read_tables_reset(gif_private_read_data_t *psDecomp)
{	unsigned uCode;
#if defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=1
	printf("Resetting input code table!\n");
#endif

	//Special codes used in the GIF spec
	psDecomp->byBitsPerCode	= psDecomp->byInitialBitsPerCode;

	//Initial size of the code and its maximum value
	psDecomp->byCurrentBits	= (BYTE)(psDecomp->byBitsPerCode+1U);

	// 0..uNPixvals-1 = Pixels, uNPixvals+0=CLEAR, uNPixvals+1=EOI
	psDecomp->uNPixvals	= (1U << psDecomp->byBitsPerCode);

	// Skip the Clear and EOI code values
	psDecomp->codeNext	= (gif_code_t)(psDecomp->uNPixvals + 2U);

#ifdef GIF_LOAD_STATS
	if (psDecomp->uReadBits!=0 &&
		GIF_LOAD_STATS>=2)
	{	printf("Resetting code table after %u (+%u) read bits, %u (+%u) read codes, %u (+%u) pixels written\n",
			psDecomp->uReadBits,psDecomp->uReadBits-psDecomp->uLastDispReadBits,
			psDecomp->uReadCodes,psDecomp->uReadCodes-psDecomp->uLastDispReadCodes,
			psDecomp->uWrittenPixels,psDecomp->uWrittenPixels-psDecomp->uLastDispWrittenPixels);
		psDecomp->uLastDispReadBits=psDecomp->uReadBits;
		psDecomp->uLastDispReadCodes=psDecomp->uReadCodes;
		psDecomp->uLastDispWrittenPixels=psDecomp->uWrittenPixels;
	}
#endif

	/* We need to start over again: */
	for (uCode = 0; uCode<NENTRIES(psDecomp->awPrefix); uCode++)
		psDecomp->awPrefix[uCode] = GIF_NO_SUCH_CODE;
}

static bool gif_private_read_data_init(gif_private_read_data_t *psDecomp)
{	BYTE bTemp;

	bTemp=0;
	if (gif_private_data_read(psDecomp,&bTemp,1,psDecomp->pDataHandle))
		return true;
	psDecomp->byInitialBitsPerCode=bTemp;	/* Read Code size from file. */

#ifdef GIF_LOAD_STATS
	psDecomp->uReadBits=psDecomp->uReadCodes=psDecomp->uWrittenPixels=0;
	psDecomp->uLastDispReadBits=psDecomp->uLastDispReadCodes=psDecomp->uLastDispWrittenPixels=0;
#endif

	gif_private_read_tables_reset(psDecomp);
	psDecomp->abyBuf[0]	= 0;	/* Input Buffer empty. */
	psDecomp->uNDecoded	= 0;	/* No pixels on the pixel stack. */
	psDecomp->codePrevious	= GIF_NO_SUCH_CODE;
	psDecomp->nDataBits	= 0;	/* No information in dwDataBits. */
	psDecomp->dwDataBits	= 0;

	return false;
}

#ifdef GIF_LOAD_STATS
static void gif_private_hexdump_data(BYTE const *pbyData,unsigned const uNBytes)
{	unsigned uLine;
	for (uLine=0 ; uLine<uNBytes ; uLine+=16)
	{	unsigned uCol,uPos;
		fprintf(stderr,"%03X:",uLine);
		for (uCol=0,uPos=uLine ; uCol<16 ; uCol++,uPos++)
		{	if (uPos<uNBytes)
			{	fprintf(stderr," %02X",pbyData[uPos]);
			} else
			{	fprintf(stderr," ..");
			}
			if (uCol==7 || uCol==15)
				fprintf(stderr," ");
		}
		for (uCol=0,uPos=uLine ; uCol<16 ; uCol++,uPos++)
		{	fprintf(stderr,"%c",(uPos<uNBytes && isprint(pbyData[uPos])) ? pbyData[uPos] : '.');
			if (uCol==7)
				fprintf(stderr," ");
		}
		fprintf(stderr,"\n");
	}
}
#endif

#ifdef GIF_LOAD_DEBUG
static void gif_private_debug_dump_buf(gif_private_read_data_t const *psDecomp)
{
	#if GIF_LOAD_DEBUG>=4
		{	unsigned uByte;
			printf("R%02X",psDecomp->abyBuf[0]);
			for (uByte=1 ; uByte<=psDecomp->abyBuf[0] ; uByte++)
				printf(" %02X",psDecomp->abyBuf[uByte]);
			printf("\n");
		}
	#else
		printf("Buffered %02X bytes\n",psDecomp->abyBuf[0]);
	#endif
}
#endif

static bool gif_private_read_next_byte(BYTE *pDest,gif_private_read_data_t *psDecomp)
{	if (psDecomp->abyBuf[0]==0)
	{	// Buffer is empty
		do
		{	if (gif_private_data_read(psDecomp,psDecomp->abyBuf,1,psDecomp->pDataHandle))
				return true;
		}
		while (psDecomp->abyBuf[0]==0);
		if (gif_private_data_read(psDecomp,psDecomp->abyBuf+1,psDecomp->abyBuf[0],psDecomp->pDataHandle))
			return true;
#ifdef GIF_LOAD_DEBUG
		gif_private_debug_dump_buf(psDecomp);
#endif
		*pDest=psDecomp->abyBuf[1];
		psDecomp->abyBuf[0]--;	// abyBuf[0]=bytes remaining in buffer
		psDecomp->abyBuf[1]=2;	// abyBuf[1]=pointer to next byte
	} else
	{	psDecomp->abyBuf[0]--;
		*pDest=psDecomp->abyBuf[psDecomp->abyBuf[1]++];
	}
	return false;
}

/******************************************************************************
*	The LZ decompression input routine:
*	This routine is responsable for the decompression of the bit stream from
*		8 bits (bytes) packets, into the real codes.
*	Returns true on error
******************************************************************************/
static bool gif_private_read_code(WORD *pCode, gif_private_read_data_t *psDecomp)
{	BYTE NextByte;
	static const WORD CodeMasks[] =
	{	0x0000, 0x0001, 0x0003, 0x0007,
		0x000f, 0x001f, 0x003f, 0x007f,
		0x00ff, 0x01ff, 0x03ff, 0x07ff,
		0x0fff
	};

	while (psDecomp->nDataBits < psDecomp->byCurrentBits)
	{	/* Needs to get more bytes from input stream for next code: */
		if (gif_private_read_next_byte(&NextByte,psDecomp))
			return true;
		psDecomp->dwDataBits |= ((unsigned long) NextByte) << psDecomp->nDataBits;
		psDecomp->nDataBits += 8;
	}

	*pCode = (WORD)(psDecomp->dwDataBits & CodeMasks[psDecomp->byCurrentBits]);

#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=2) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
	if (GIF_LOAD_STATS>=3)
#endif
	printf("Read %u bits as %03X: ",psDecomp->byCurrentBits,*pCode);
#endif

#ifdef GIF_LOAD_STATS
	psDecomp->uReadBits+=psDecomp->byCurrentBits;
	psDecomp->uReadCodes++;
#endif

	psDecomp->dwDataBits >>= psDecomp->byCurrentBits;
	psDecomp->nDataBits = (BYTE)(psDecomp->nDataBits-psDecomp->byCurrentBits);

	/* If code can't fit into byCurrentBits bits, must raise its size.
	 * Note: Codes >=GIF_LZ_NCODES are used for special signaling.
	 */
	if (psDecomp->codeNext>=GIF_LZ_NCODES)
	{
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=3) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS>=4)
#endif
			printf("At code size limit of %u bits\n",psDecomp->byCurrentBits);
#endif
		psDecomp->codeNext++;
	} else
	if (++psDecomp->codeNext > (1U << psDecomp->byCurrentBits) &&
		psDecomp->byCurrentBits < GIF_LZ_BITS)
	{	psDecomp->byCurrentBits++;
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=3) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS>=4)
#endif
			printf("Increased code size to %u bits\n",psDecomp->byCurrentBits);
#endif
	}
	return false;
}

static unsigned gif_private_read_prefix(gif_private_read_data_t const *psDecomp, unsigned uStartCode)
{	unsigned uLoop;
	unsigned uCode=uStartCode;
	for (uLoop=0 ; uLoop<GIF_LZ_NCODES ; uLoop++)
	{
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=3) || defined(GIF_LOAD_STATS)
		if (uLoop==0)
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS>=4)
#endif
			printf("Chain=");
		{
#ifdef GIF_LOAD_STATS
			if (GIF_LOAD_STATS>=4)
#endif
				printf("%03X",uCode);
			if (uCode<psDecomp->uNPixvals)
			{
#ifdef GIF_LOAD_STATS
				if (GIF_LOAD_STATS>=4)
#endif
					printf("\n");
				return uCode;
			}
			if (uCode>=GIF_LZ_NCODES)
			{
#ifdef GIF_LOAD_STATS
				if (GIF_LOAD_STATS>=4)
#endif
					printf("\n");
				break;
			}
#ifdef GIF_LOAD_STATS
			if (GIF_LOAD_STATS>=4)
#endif
				printf("(%02X)->",psDecomp->abySuffix[uCode]);
		}
#else
		if (uCode<psDecomp->uNPixvals) return uCode;
		if (uCode>=GIF_LZ_NCODES) break;
#endif
		uCode = psDecomp->awPrefix[uCode];
	}
	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Code table overflow %u codes, starting with %03X ending with %03X\n",uLoop+1,uStartCode,uCode);
	((gif_private_read_data_t *)psDecomp)->uNWarnings++;
	return GIF_NO_SUCH_CODE;
}

static bool gif_private_read_data(gif_file_t *psGifInfo, BYTE *pDest, unsigned uNPixels)
{	unsigned uDone=0;
#if defined(GIF_LOAD_DEBUG) || defined(_DEBUG)
	gif_image_t const * const psImage=&psGifInfo->psImages[psGifInfo->uNImages-1];
	unsigned const uNColours=(psImage->image.bLocalColourTable) ? GIF_NCOLOURS(psImage->image) : GIF_NCOLOURS(psGifInfo->screen);
#endif
	gif_private_read_data_t *psDecomp= (gif_private_read_data_t *)psGifInfo->pPrivate;
	WORD wThisCode;
	bool bHadInitialReset=false;

#if 0 // All read in one go, so this isn't needed
	// Shortcut to full loop!
	while (uDone<uNPixels && psDecomp->uNDecoded>0)
	{	pDest[uDone++]=psDecomp->abyDecoded[--psDecomp->uNDecoded];
		GIF_ASSERT(psDecomp->abyDecoded[psDecomp->uNDecoded]<uNColours);
#ifdef GIF_LOAD_STATS
		psDecomp->uWrittenPixels++;
#endif
#if defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4
		printf("Out %5u: %02X\n",uDone,pDest[uDone-1]);
#endif
	}
#endif

	// Not had any overruns yet
	psGifInfo->sOptions.bNoOverrun=true;

	// While pixels to do, decode LZW info
	while (uDone < uNPixels)
	{
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS>=3)
			printf("Pixel %u = %u bits, codes=0x%X\n",
				psDecomp->uWrittenPixels,psDecomp->uReadBits,psDecomp->uReadCodes);
#endif

		if (gif_private_read_code(&wThisCode, psDecomp))
		{	gif_free(psGifInfo);
			return true;
		}

#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG<2) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS==2)
#endif
			printf("%03X.",wThisCode);
#endif
		if (wThisCode == (psDecomp->uNPixvals+1))	// uNPixvals+1 = EOI
		{	gif_free(psGifInfo);
			fprintf(stderr,"Error: EOI unexpected\n");
			return true;
		} else

		if (wThisCode == (psDecomp->uNPixvals+0))	// uNPixvals+0 = Clear
		{	gif_private_read_tables_reset(psDecomp);
			psDecomp->codePrevious = GIF_NO_SUCH_CODE;
			if (uDone==0) bHadInitialReset=true;
		} else

		{
			if (uDone==0 && !bHadInitialReset)
			{	psGifInfo->sOptions.bMissingInitialResets=true;
			}
			/*
			 * Its regular code -
			 * If in pixel range simply add it to output stream,
			 * otherwise trace to codes linked list until the prefix is in pixel range
			 */
			if (wThisCode < psDecomp->uNPixvals)
			{	/* This is simple - its pixel scalar, so add it to output:   */
				GIF_ASSERT(wThisCode<uNColours);
				pDest[uDone++] = (BYTE)wThisCode;
#ifdef GIF_LOAD_STATS
				psDecomp->uWrittenPixels++;
#endif
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
				if (GIF_LOAD_STATS>=5)
#endif
					printf("Out %5u: %02X\n",uDone,pDest[uDone-1]);
#endif
			} else
			{	unsigned uCurrentPrefix=GIF_NO_SUCH_CODE;
				GIF_ASSERT(psDecomp->uNDecoded==0);

				/*
				 * Its a code to needed to be traced:
				 *	trace the linked list until the prefix is a pixel,
				 *	while pushing the suffix pixels on our stack.
				 *	If we're done, pop the stack in reverse order to output.
				 */
				GIF_ASSERT(wThisCode<NENTRIES(psDecomp->awPrefix));
				if (psDecomp->awPrefix[wThisCode] == GIF_NO_SUCH_CODE)
				{	// It's not an existing code
					GIF_ASSERT(psDecomp->codeNext>=2);
					GIF_ASSERT(wThisCode<NENTRIES(psDecomp->abySuffix));
					GIF_ASSERT(psDecomp->uNDecoded<NENTRIES(psDecomp->abyDecoded));

					/* Only allowed if codeCurrent is exactly the running code:
					 * In that case codeCurrent = XXXCode, codeCurrent or the
					 * prefix code is last code and the suffix char is
					 * exactly the prefix of last code!
					 */
					if (wThisCode == psDecomp->codeNext - 2)
					{	uCurrentPrefix = psDecomp->codePrevious;
						psDecomp->abySuffix[wThisCode] =
							psDecomp->abyDecoded[psDecomp->uNDecoded] = (BYTE)gif_private_read_prefix(psDecomp, psDecomp->codePrevious);
						GIF_ASSERT(psDecomp->abyDecoded[psDecomp->uNDecoded]<uNColours);
						psDecomp->uNDecoded++;
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
						if (GIF_LOAD_STATS>=5)
#endif
							printf("Adding [%u]=%02X\n",psDecomp->uNDecoded-1,psDecomp->abyDecoded[psDecomp->uNDecoded-1]);
#endif
						GIF_ASSERT(psDecomp->uNDecoded<=NENTRIES(psDecomp->abyDecoded));
					} else
					{	fprintf(stderr,"Error: Decompression error (Code %03X unallocated, last=%03X, next=%03X, %u bits)\n",wThisCode,psDecomp->codePrevious,psDecomp->codeNext,psDecomp->byCurrentBits);
						//gif_private_read_dump_used_codes(psDecomp);
						gif_free(psGifInfo);
						return true;
					}
				} else
				{	// It's an existing code
					uCurrentPrefix = wThisCode;
				}

				/* Now (if image is O.K.) we should not get a NO_SUCH_CODE
				 * During the trace. As we might loop forever, in case of
				 * defective image, we count the number of loops we trace
				 * and stop if we got LZ_MAX_CODE. obviously we can not
				 * loop more than that.
				 */
				{	unsigned uStartCode=uCurrentPrefix;
					unsigned uCode;
					for (uCode=0 ; uCode<GIF_LZ_NCODES ; uCode++)
					{	if (uCurrentPrefix < psDecomp->uNPixvals ||		// Reached end
							uCurrentPrefix >=GIF_LZ_NCODES) break;	// got an illegal code
						GIF_ASSERT(psDecomp->uNDecoded<NENTRIES(psDecomp->abyDecoded));
						GIF_ASSERT(psDecomp->abySuffix[uCurrentPrefix]<uNColours);
						psDecomp->abyDecoded[psDecomp->uNDecoded++] = psDecomp->abySuffix[uCurrentPrefix];
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
						if (GIF_LOAD_STATS>=5)
#endif
							printf("Adding [%u]=%02X for %03X\n",psDecomp->uNDecoded-1,psDecomp->abyDecoded[psDecomp->uNDecoded-1],uCurrentPrefix);
#endif
						// Next link
						uCurrentPrefix = psDecomp->awPrefix[uCurrentPrefix];
					}
					if (uCode >= GIF_LZ_NCODES || uCurrentPrefix >= psDecomp->uNPixvals)
					{	gif_free(psGifInfo);
						fprintf(stderr,"Error: Decompression error - %u links starting with %03X, ending with %03X\n",uCode,uStartCode,uCurrentPrefix);
						return true;
					}
				}
				/* Push the last character on stack: */
				GIF_ASSERT(psDecomp->uNDecoded<NENTRIES(psDecomp->abyDecoded));
				GIF_ASSERT(uCurrentPrefix<uNColours);
				psDecomp->abyDecoded[psDecomp->uNDecoded++] = (BYTE)uCurrentPrefix;
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
				if (GIF_LOAD_STATS>=5)
#endif
					printf("Adding [%u]=%02X\n",psDecomp->uNDecoded-1,psDecomp->abyDecoded[psDecomp->uNDecoded-1]);
#endif
				// Shortcut to full loop!
				while (uDone<uNPixels && psDecomp->uNDecoded>0)
				{	pDest[uDone++]=psDecomp->abyDecoded[--psDecomp->uNDecoded];
					GIF_ASSERT(psDecomp->abyDecoded[psDecomp->uNDecoded]<uNColours);
#ifdef GIF_LOAD_STATS
					psDecomp->uWrittenPixels++;
#endif
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
					if (GIF_LOAD_STATS>=5)
#endif
						printf("Out %5u: %02X\n",uDone,pDest[uDone-1]);
#endif
				}
			}

			if (psDecomp->codePrevious != GIF_NO_SUCH_CODE)
			{	// Check code is in range
				GIF_ASSERT(psDecomp->codeNext>=(psDecomp->uNPixvals+2));
				if ((psDecomp->codeNext-2)>=(int)NENTRIES(psDecomp->awPrefix) ||
					(psDecomp->codeNext-2)>=(int)NENTRIES(psDecomp->abySuffix))
				{	// Table has overrun, this is allowed in Gif89a
					//fprintf(stderr,"Error: Code %03X overflowed\n",psDecomp->codeNext);
					//gif_private_read_dump_used_codes(psDecomp);
					psGifInfo->sOptions.bNoOverrun=false;
					continue;
				} else
				{	// Assign the code
					unsigned uToAssign,uChainedCode;
					if (wThisCode == psDecomp->codeNext - 2 || wThisCode == psDecomp->codeNext - 1)
					{	/* Only allowed if codeCurrent is exactly the running code:gifinfo
						 * In that case codeCurrent = XXXCode, codeCurrent or the
						 * prefix code is last code and the suffix char is
						 * exactly the prefix of last code!
						 */
						uToAssign=wThisCode;
						uChainedCode=psDecomp->codePrevious;
					} else
					{	uToAssign=psDecomp->codeNext - 2;
						uChainedCode=wThisCode;
					}

					if (psDecomp->awPrefix[uToAssign] == GIF_NO_SUCH_CODE)
					{	// Link the chain
						psDecomp->awPrefix[uToAssign] = (WORD)psDecomp->codePrevious;
						psDecomp->abySuffix[uToAssign] = (BYTE)gif_private_read_prefix(psDecomp, uChainedCode);
#if (defined(GIF_LOAD_DEBUG) && GIF_LOAD_DEBUG>=4) || defined(GIF_LOAD_STATS)
#ifdef GIF_LOAD_STATS
						if (GIF_LOAD_STATS>=5)
#endif
							printf("awPrefix[%03X]=%03X, abySuffix[%03X]=%02X\n",
								uToAssign,psDecomp->awPrefix[uToAssign],uToAssign,psDecomp->abySuffix[uToAssign]);
#endif
					} else
					{	fprintf(stderr,"Error: Code %03X already assigned\n",uToAssign);
						return true;
					}
				}
			}
			psDecomp->codePrevious = wThisCode;
		}
	}

	//gif_private_read_dump_used_codes(psDecomp);
	return false;
}


// Rearrange image so that pixels are in the gif file order
// Only needed after switching interlaced mode
static bool gif_private_rearrange_image(gif_image_t *psImage)
{	unsigned uLine;
	BYTE **pLineData;

	// Doesn't matter if it's interlaced or not
	if (psImage->image.height<=1 || psImage->image.width==0) return false;

	// Double check lines are in the correct order
	//GIF_ASSERT(gif_private_interlaced_correct(psImage));

	// Move actual pixel data to one side
	pLineData=psImage->pLines;
	psImage->pLines=NULL;

	// Allocate new data area
	if (gif_allocate_image_memory(psImage))
	{	// If it failed, but real data back, and fail
		psImage->pLines=pLineData;
		return true;
	}

	// Copy line data into correct order
	for (uLine=0 ; uLine<psImage->image.height ; uLine++)
	{	memcpy(psImage->pLines[uLine],pLineData[uLine],psImage->image.width);
	}
	free(pLineData);

	return false;
}

static bool gif_private_read_image(gif_file_t *psGifInfo, gif_image_t *psImage)
{	gif_private_read_data_t *psDecomp= (gif_private_read_data_t *)psGifInfo->pPrivate;
	unsigned uLine;
	BYTE byTerminator;
	bool bGotEOI=false;

	GIF_ASSERT((unsigned)(psImage-psGifInfo->psImages)<psGifInfo->uNImages);

	if (gif_allocate_image_memory(psImage) ||
		gif_private_read_data_init(psDecomp))
	{	gif_free(psGifInfo);
		return true;
	}

	// Read pixels from file
	if (gif_private_read_data(psGifInfo, psImage->pLines[0], psImage->image.width*psImage->image.height))
	{	gif_free(psGifInfo);
		fprintf(stderr,"Error: Image %u - compressed data illegal\n",(unsigned)(psImage-psGifInfo->psImages));
		return true;
	}

	for (uLine=0 ; uLine<psImage->image.height ; uLine++)
	{	const unsigned uNColours=
			(2U<<(psImage->image.bLocalColourTable ? psImage->image.bitsPerPixelMinus1 : psGifInfo->screen.bitsPerPixelMinus1));
		unsigned uColumn;
		for (uColumn=0; uColumn<psImage->image.width ; uColumn++)
		if (psImage->pLines[uLine][uColumn]>=uNColours)
		{	fprintf(stderr,"Error: Image %u - Colour %u in %u colour image at %u,%u (%u)\n",
				psImage-psGifInfo->psImages,
				psImage->pLines[uLine][uColumn],uNColours,
				uColumn,uLine,gif_private_lookup_interlaced_line(&psImage->image,uLine));
		}
	}

	// Check if we had any stacked bytes
	if (psDecomp->uNDecoded!=0)
	{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - Excess data, %u decoded bytes!\n",psImage-psGifInfo->psImages,psDecomp->uNDecoded);
#ifdef GIF_LOAD_STATS
		if (GIF_LOAD_STATS==2)
		{	unsigned uByte;
			for (uByte=0 ; uByte<psDecomp->uNDecoded ; uByte++)
				printf(" %0X",psDecomp->abyDecoded[uByte]);
			printf("\n");
		}
#endif
		psDecomp->uNWarnings++;
	}

	// Scan spare code bits for an EOI
	while (!bGotEOI && (psDecomp->abyBuf[0]*8U+psDecomp->nDataBits)>=psDecomp->byCurrentBits)
	{	WORD wEOI=0xFFFF;
		if (gif_private_read_code(&wEOI, psDecomp)) return true;
		if (wEOI==(psDecomp->uNPixvals+1))	// uNPixvals+1==EOI
		{	bGotEOI=true;
		} else
		{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - Expected EOI of %X, got %X\n",psImage-psGifInfo->psImages,(psDecomp->uNPixvals+1),wEOI);
			psDecomp->uNWarnings++;
		}
	}

	// Try to read a terminator
	if (gif_private_data_read(psDecomp,&byTerminator,1,psDecomp->pDataHandle)) return true;

	if (byTerminator!=0)
	{	// Non zero, if we've not had an EOI check to see if it's a data block with one in
		if (!bGotEOI && ((psDecomp->abyBuf[0]+byTerminator)*8+psDecomp->nDataBits)<(psDecomp->byCurrentBits+8))
		{	WORD wEOI=0xFFFF;
			BYTE byNextByte;
			// We've got the right number of extra bytes for an EOI
			while (psDecomp->abyBuf[0]>0)
			{	if (gif_private_read_next_byte(&byNextByte,psDecomp))
					return true;
				psDecomp->dwDataBits |= ((unsigned long) byNextByte) << psDecomp->nDataBits;
				psDecomp->nDataBits += 8;
			}
			// Read the terminator bytes as well
			for ( ; byTerminator>0 ; byTerminator--)
			{	if (gif_private_data_read(psDecomp,&byNextByte,1,psDecomp->pDataHandle)) return true;
				psDecomp->dwDataBits |= ((unsigned long) byNextByte) << psDecomp->nDataBits;
				psDecomp->nDataBits += 8;
			}
			// Now read the code
			if (gif_private_read_code(&wEOI, psDecomp)) return true;
			// Check it
			if (wEOI==(psDecomp->uNPixvals+1))	// uNPixvals+1==EOI
			{	bGotEOI=true;
				// If we found it, hopefully we've got a terminator next
				if (gif_private_data_read(psDecomp,&byTerminator,1,psDecomp->pDataHandle)) return true;
			} else
			{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - Expected EOI of %X, got %X\n",psImage-psGifInfo->psImages,(psDecomp->uNPixvals+1),wEOI);
				psDecomp->uNWarnings++;
			}
		}
		if (byTerminator!=0)
		{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - Terminator corrupted (got %02X)\n",psImage-psGifInfo->psImages,byTerminator);
			psDecomp->uNWarnings++;
		}
	}
	if (!bGotEOI)
	{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - No EOI found!\n",psImage-psGifInfo->psImages);
		psDecomp->uNWarnings++;
	}
	if ((psDecomp->abyBuf[0]*8U+psDecomp->nDataBits)>=8)
	{	if (!psDecomp->bQuiet) fprintf(stderr,"Warning: Image %u - Excess data, %u spare bits (vs %u codelen)!\n",
				psImage-psGifInfo->psImages,psDecomp->abyBuf[0]*8U+psDecomp->nDataBits,psDecomp->byCurrentBits);
		psDecomp->uNWarnings++;
	}
#ifdef GIF_LOAD_STATS
	printf("EOI %u after %u (+%u) bits, %u (+%u) codes, %u (+%u) pixels\n",
			psImage-psGifInfo->psImages,
			psDecomp->uReadBits,psDecomp->uReadBits-psDecomp->uLastDispReadBits,
			psDecomp->uReadCodes,psDecomp->uReadCodes-psDecomp->uLastDispReadCodes,
			psDecomp->uWrittenPixels,psDecomp->uWrittenPixels-psDecomp->uLastDispWrittenPixels);
#endif

	return false;
}

// Try to find signature in unused palette entries
static bool gif_private_palette_signature_check(gif_file_t *psGifInfo)
{	gif_image_t const *psImage;
	unsigned uImage;
	unsigned uPos;
	unsigned uColour;
	unsigned uLen;
	unsigned uNColours;
	BYTE const *pbyPixel;
	bool bFound=false;

	char string1[4]="SG";			// + 0..9
	char string2[7]="SMGIF";		// + 0..9
	char string3[10]="SMALLGIF";	// + 0..9
	char string4[13]="SMALLGIF.COM";// With varying upper and lower case
	char *pszString[4];
	bool abUsed[256];

	pszString[0]=string1;
	pszString[1]=string2;
	pszString[2]=string3;
	pszString[3]=string4;

	// This is the way the compression level is added
	//		pszString[0][2]=pszString[1][5]=pszString[2][8]=(char)('0'+uOptimisation);
	//		if (uOptimisation<9) strlwr(pszString[3]+uOptimisation);

	if (psGifInfo->screen.bGlobalColourTable)
	{	// Search used global colours
		memset(abUsed,0,sizeof(abUsed));
		for (uImage=0, psImage=psGifInfo->psImages ; uImage<psGifInfo->uNImages ; uImage++,psImage++)
		if (!psImage->image.bLocalColourTable)
		{	// Find transcol, or 999 if none
			unsigned const uTransCol=(psImage->bHaveControl && psImage->control.bTransparent) ? psImage->control.byTransparentColour : 999U;
			unsigned const uNPixels=psImage->image.height*psImage->image.width;
			// Scan image colour entries, and mark in global table
			for (uPos=0,pbyPixel=psImage->pLines[0] ; uPos<uNPixels ; uPos++,pbyPixel++)
				if (*pbyPixel!=uTransCol) abUsed[*pbyPixel]=true;
		}

		// Try to find blocks that are unused
		uNColours=GIF_NCOLOURS(psGifInfo->screen);
		for (uColour=0 ; uColour<uNColours ; uColour++)
		{	for (uLen=0 ; uLen<5 && (uColour+uLen)<uNColours ; uLen++)
				if (abUsed[uColour+uLen]) break;
			if (uLen>0)
			{	// Found block, check string
				char const *pszToCheck=pszString[uLen>3 ? 3 : (uLen-1)];
				char const *pszPalettePos=(char *)(psGifInfo->psColourTable+uColour);
				unsigned const uToCheck=strlen(pszToCheck);
				GIF_ASSERT(uToCheck<=(3*uLen));
				GIF_ASSERT((uColour+uLen)<=uNColours);
				if (uLen<=3)
				{	if (memcmp(pszPalettePos,pszToCheck,uToCheck)==0 && pszPalettePos[uToCheck]>='0' && pszPalettePos[uToCheck]<='9')
					{	// Found it
						psGifInfo->sOptions.byCompressionLevel=(BYTE)(pszPalettePos[uToCheck]-'0');
						return true;
					}
				} else
				if (strnicmp(pszPalettePos,pszToCheck,uToCheck)==0)
				{	// Is a long string, check where the first lower case is
					for (psGifInfo->sOptions.byCompressionLevel=0 ; psGifInfo->sOptions.byCompressionLevel<9 ; psGifInfo->sOptions.byCompressionLevel++)
						if (islower(pszPalettePos[psGifInfo->sOptions.byCompressionLevel])) break;
					return true;
				}
				uColour+=uLen;
			}
		}
	}

	for (uImage=0, psImage=psGifInfo->psImages ; uImage<psGifInfo->uNImages ; uImage++,psImage++)
	if (psImage->image.bLocalColourTable)
	{	// Find transcol, or 999 if none
		unsigned const uTransCol=(psImage->bHaveControl && psImage->control.bTransparent) ? psImage->control.byTransparentColour : 999U;
		unsigned const uNPixels=psImage->image.height*psImage->image.width;
		memset(abUsed,0,sizeof(abUsed));
		// Scan image colour entries, and mark in local table
		for (uPos=0,pbyPixel=psImage->pLines[0] ; uPos<uNPixels ; uPos++,pbyPixel++)
			if (*pbyPixel!=uTransCol) abUsed[*pbyPixel]=true;

		// Try to find block that are unused
		uNColours=GIF_NCOLOURS(psImage->image);
		for (uColour=0 ; uColour<uNColours ; uColour++)
		{	for (uLen=0 ; uLen<5 && (uColour+uLen)<uNColours ; uLen++)
				if (abUsed[uColour+uLen]) break;
			if (uLen>0)
			{	// Found block, check string
				char const *pszToCheck=pszString[uLen>3 ? 3 : (uLen-1)];
				char const *pszPalettePos=(char *)(psImage->psColourTable+uColour);
				unsigned const uToCheck=strlen(pszToCheck);
				GIF_ASSERT(uToCheck<=(3*uLen));
				GIF_ASSERT((uColour+uLen)<=uNColours);
				if (uLen<3)
				{	if (memcmp(pszPalettePos,pszToCheck,uToCheck)==0 && pszPalettePos[uToCheck]>='0' && pszPalettePos[uToCheck]<='9')
					{	// Found it
						psGifInfo->sOptions.byCompressionLevel=(BYTE)(pszPalettePos[uToCheck]-'0');
						return true;
					}
				} else
				if (strnicmp(pszPalettePos,pszToCheck,uToCheck)==0)
				{	// Is a long string, check where the first lower case is
					for (psGifInfo->sOptions.byCompressionLevel=0 ; psGifInfo->sOptions.byCompressionLevel<9 ; psGifInfo->sOptions.byCompressionLevel++)
						if (islower(pszPalettePos[psGifInfo->sOptions.byCompressionLevel])) break;
					return true;
				}
				uColour+=uLen;
			}
		}
	}
	return bFound;
}

// Read a gif file, using supplied read function
static bool gif_private_read_images(
		gif_file_t *psGifInfo,
		bool	bQuiet,
		bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle),
		void	*pDataHandle)
{	gif_private_read_data_t sDecomp;
	gif_start_t gifHeader;

	memset(psGifInfo,0,sizeof(*psGifInfo));
	psGifInfo->pPrivate=(void *)&sDecomp;
	GIF_ASSERT(read_data!=NULL);
	sDecomp.read_data=read_data;
	sDecomp.pDataHandle=pDataHandle;
	sDecomp.uReadBytes=0;
	sDecomp.bQuiet=bQuiet;
	sDecomp.uNWarnings=0;

	// Set invalid compression mode, to confirm if we find something
	psGifInfo->sOptions.byCompressionLevel=0xFF;

	memset(&gifHeader,0,sizeof(gifHeader));

	if (gif_private_data_read(&sDecomp,(BYTE*)&gifHeader,sizeof(gifHeader),pDataHandle) ||
		gifHeader.sig_G!='G' ||
		gifHeader.sig_I!='I' ||
		gifHeader.sig_F!='F' ||
		(memcmp(gifHeader.ver,"87a",3)!=0 && memcmp(gifHeader.ver,"89a",3)!=0) )
	{	fprintf(stderr,"Bad GIF header\n");
		return true;
	}

	// Start gathering info about the file
	if (gif_private_data_read(&sDecomp,(BYTE *)&psGifInfo->screen,sizeof(psGifInfo->screen),pDataHandle) ||
			psGifInfo->screen.height==0 ||
			psGifInfo->screen.width==0)
	{	fprintf(stderr,"Bad GIF screen info\n");
		return true;
	}
#ifdef IS_BIG_ENDIAN
	psGifInfo->screen.width=BE_WORD(psGifInfo->screen.width);
	psGifInfo->screen.height=BE_WORD(psGifInfo->screen.height);
	{	BYTE const byBitVals= ((BYTE *)&psGifInfo->screen)[4];
		psGifInfo->screen.bitsPerPixelMinus1=(byBitVals)&7;
		psGifInfo->screen.sort=(byBitVals>3)&1;
		psGifInfo->screen.bitsPerColourValMinus1=(byBitVals>>4)&7;
		psGifInfo->screen.bGlobalColourTable=(byBitVals>>7)&1;
	}
#endif

	psGifInfo->psColourTable=NULL;
	if (psGifInfo->screen.bGlobalColourTable)
	{
#ifdef GIF_LOAD_STATS
		printf("Reading global colour table with %u colours\n",GIF_NCOLOURS(psGifInfo->screen));
#endif
		if (gif_private_read_colour_table(&psGifInfo->psColourTable,&psGifInfo->uColourTableBytes,psGifInfo->screen.bitsPerPixelMinus1,&sDecomp))
		{	return true;
		}
	}

	for(;;)
	{	gif_image_t *psImage=(psGifInfo->uNImages==0) ? NULL : &psGifInfo->psImages[psGifInfo->uNImages-1];
		BYTE bLen;
		bLen=0;
		if (gif_private_data_read(&sDecomp,&bLen,1,pDataHandle))
		{	// Assume end of file
			if (!bQuiet) fprintf(stderr,"Warning: Unexpected end of image data!\n");
			break;
		}
		if (bLen==0x3B) break;
		switch (bLen)
		{	case 0x2C:	// Image identifier
				if (psImage==NULL || psImage->pLines!=NULL)
				{	if ((psImage=gif_add_image_space(psGifInfo))==NULL)
						return true;
				}
				if (gif_private_data_read(&sDecomp,(BYTE *)&psImage->image,sizeof(psImage->image),pDataHandle))
					return true;
#ifdef IS_BIG_ENDIAN
				psImage->image.left=BE_WORD(psImage->image.left);
				psImage->image.top=BE_WORD(psImage->image.top);
				psImage->image.width=BE_WORD(psImage->image.width);
				psImage->image.height=BE_WORD(psImage->image.height);
				{	BYTE const byBitVals=((BYTE *)&psImage->image)[8];
					psImage->image.bitsPerPixelMinus1=byBitVals&7;
					psImage->image.Reserved1=0;
					psImage->image.bSorted=(byBitVals>>5)&1;
					psImage->image.bInterlaced=(byBitVals>>6)&1;
					psImage->image.bLocalColourTable=(byBitVals>>7)&1;
				}
#endif
				if ( ((psImage->image.left+psImage->image.width)>psGifInfo->screen.width) ||
					((psImage->image.top+psImage->image.height)>psGifInfo->screen.height))
				{	// Image is outside of current screen info
					if (!bQuiet)
						fprintf(stderr,"Warning: Growing (%d,%d) screen to allow for (%d,%d)-(%d,%d) image\n",
							psGifInfo->screen.width,psGifInfo->screen.height,psImage->image.left,psImage->image.top,psImage->image.width,psImage->image.height);
					if ((psImage->image.left+psImage->image.width)>psGifInfo->screen.width)
					{	psGifInfo->screen.width=(WORD)(psImage->image.left+psImage->image.width);
					}
					if ((psImage->image.top+psImage->image.height)>psGifInfo->screen.height)
					{	psGifInfo->screen.height=(WORD)(psImage->image.top+psImage->image.height);
					}
				}
				psImage->psColourTable=NULL;
				if (psImage->image.bLocalColourTable)
				{
#ifdef GIF_LOAD_STATS
					printf("Reading local colour table with %u colours\n",GIF_NCOLOURS(psImage->image));
#endif
					if (gif_private_read_colour_table(&psImage->psColourTable,&psImage->uColourTableBytes,psImage->image.bitsPerPixelMinus1,&sDecomp))
					{	return true;
					}
				}
#ifdef GIF_LOAD_STATS
				if (GIF_LOAD_STATS>=2)
					printf("Reading image %u (%u,%u)-(%u,%u)\n",
						psGifInfo->uNImages-1,
						psImage->image.left,psImage->image.top,
						psImage->image.left+psImage->image.width,
						psImage->image.top+psImage->image.height);
#endif
				if (gif_private_read_image(psGifInfo,psImage))
				{	return true;
				}
				break;
			case 0x21:	// Extension
				bLen=0;
				if (gif_private_data_read(&sDecomp,&bLen,1,pDataHandle))
				{	return true;
				}
				switch (bLen)
				{	case 0xF9:	// Extension F9 = control block
						if (psImage==NULL || psImage->pLines!=NULL)
						{	if ((psImage=gif_add_image_space(psGifInfo))==NULL)
								return true;
						} else
						if (psImage->bHaveControl)
						{	if (!bQuiet) fprintf(stderr,"Warning: Multiple control blocks, earlier blocks ignored\n");
							psImage->bHaveControl=false;
							memset(&psImage->control,0,sizeof(psImage->control));
						}
						for (;;)
						{	if (gif_private_data_read(&sDecomp,&bLen,1,pDataHandle))
							{	return true;
							}
							if (bLen==0) break;
							if (psImage->bHaveControl ||
								bLen!=sizeof(psImage->control) ||
								gif_private_data_read(&sDecomp,(BYTE *)&psImage->control,sizeof(psImage->control),pDataHandle))
							{	fprintf(stderr,"Error: Bad control block\n");
								return true;
							}
#ifdef IS_BIG_ENDIAN
							psImage->control.delaytime=BE_WORD(psImage->control.delaytime);
							{	BYTE const byBitVals=((BYTE *)&psImage->control)[0];
								psImage->control.bTransparent=(byBitVals)&1;
								psImage->control.bUserinput=(byBitVals>>1)&1;
								psImage->control.disposal=(byBitVals>>2)&7;
								psImage->control.Reserved=0;
							}
#endif
							psImage->bHaveControl=true;
						}
#ifdef GIF_LOAD_STATS
						if (GIF_LOAD_STATS>=2)
						{	printf("Read control block for image %u\n",psGifInfo->uNImages-1);
							if (GIF_LOAD_STATS>=5)
							{	printf("T=%u, U=%u, D=%u, R=%u, del=%u, tc=%02X\n",
									psImage->control.bTransparent&1,psImage->control.bUserinput&1,psImage->control.disposal&3,psImage->control.Reserved&7,psImage->control.delaytime,psImage->control.byTransparentColour);
							}
						}
#endif
						break;
					case 0xFF:	// Application block, maybe netscape loop count block
						{	gif_ext_application_t block;
							enum
							{	e_getBlock,e_skipData,e_getData
							}  eState=e_getBlock;
							for (;;)
							{	if (gif_private_data_read(&sDecomp,&bLen,1,pDataHandle))
								{	return true;
								}
								if (bLen==0) break;
								if (eState==e_getBlock)
								{	if (bLen!=sizeof(block))
									{	// Block size is wrong
										if (!sDecomp.bQuiet) fprintf(stderr,"Warning: Bad control block, ID size should be %u is %u\n",sizeof(block),bLen);
										if (bLen<sizeof(block))
										{	// Clear block and read number of bytes that we've been told are there
											memset(&block,0,sizeof(block));
											if (gif_private_data_read(&sDecomp,(BYTE *)&block,bLen,pDataHandle))
											{	return true;
											}
										} else
										{	// Read block then skip rest
											if (gif_private_data_read(&sDecomp,(BYTE *)&block,sizeof(block),pDataHandle))
											{	return true;
											}
											while (bLen<sizeof(block))
											{	BYTE bTemp;
												if (gif_private_data_read(&sDecomp,&bTemp,1,pDataHandle))
												{	return true;
												}
												bLen++;
											}
										}
										// Read
									} else
									{	// Read block normally
										if (gif_private_data_read(&sDecomp,(BYTE *)&block,sizeof(block),pDataHandle))
										{	return true;
										}
									}
									if (memcmp(&block,GIF_NETSCAPE_ID,sizeof(block))!=0)
									{	eState=e_skipData;
										if (!sDecomp.bQuiet) fprintf(stderr,"Warning: Skipping %c%c%c%c%c%c%c%c.%c%c%c application block\n",
											block.application[0],block.application[1],block.application[2],block.application[3],block.application[4],block.application[5],block.application[6],block.application[7],
											block.applicationcode[0],block.applicationcode[1],block.applicationcode[2]);
										sDecomp.uNWarnings++;
									} else
									{	eState=e_getData;
										if (psImage && psImage->bHaveNetscapeLoop)
										{	if (!sDecomp.bQuiet) fprintf(stderr,"Warning: Multiple Netscape blocks (earlier removed)\n");
											psImage->bHaveNetscapeLoop=false;
										}
										if (psImage==NULL || psImage->pLines!=NULL)
										{	if ((psImage=gif_add_image_space(psGifInfo))==NULL)
												return true;
										}
									}
								} else
								if (eState==e_getData)
								{	BYTE bExtensionID;
									if (gif_private_data_read(&sDecomp,&bExtensionID,1,pDataHandle) ||
										bExtensionID!=1 ||
										gif_private_data_read(&sDecomp,(BYTE *)&psImage->wNumLoops,sizeof(psImage->wNumLoops),pDataHandle))
									{	fprintf(stderr,"Error: Bad Netscape block data\n");
										return true;
									}
									psImage->bHaveNetscapeLoop=true;
									eState=e_skipData;
								} else
								if (eState==e_skipData)
								{	if (!sDecomp.bQuiet) fprintf(stderr,"Warning: Skipping %u bytes of application data\n",bLen);
									sDecomp.uNWarnings++;
									// Read into unused buffer
									if (gif_private_data_read(&sDecomp,sDecomp.abyBuf,bLen,pDataHandle))
									{	return true;
									}
								}
							}
						}
						break;
					default:
						{	BYTE const byBlockType=bLen;
							bool const bIsComment=SET_BOOL(byBlockType==0xFE);
							unsigned uNSkipped=2;
							for (;;)
							{	if (gif_private_data_read(&sDecomp,&bLen,1,pDataHandle))
								{	return true;
								}
								uNSkipped++;
								if (bLen==0) break;
								if (gif_private_data_read(&sDecomp,sDecomp.abyBuf,bLen,pDataHandle))
								{	return true;
								}
								uNSkipped+=bLen;
								sDecomp.abyBuf[bLen]=0;
							}
							// Not a comment, or not one of ours
							if (bIsComment && memcmp(sDecomp.abyBuf,"RNW",4)==0)
							{	psGifInfo->sOptions.byCompressionLevel=1;
							} else
							if (bIsComment && sDecomp.abyBuf[0]=='S' && sDecomp.abyBuf[1]=='G' && sDecomp.abyBuf[2]=='\0')
							{	psGifInfo->sOptions.byCompressionLevel=9;
							} else
							if (bIsComment && sDecomp.abyBuf[0]=='S' && sDecomp.abyBuf[1]=='G' && sDecomp.abyBuf[2]>='0' && sDecomp.abyBuf[2]<='9' && sDecomp.abyBuf[3]=='\0')
							{	psGifInfo->sOptions.byCompressionLevel=(BYTE)(sDecomp.abyBuf[2]-'0');
							} else
							{	if (!sDecomp.bQuiet)
								{	fprintf(stderr,"Warning: Skipping %u bytes block of type 0x%X%s\n",uNSkipped,byBlockType,(bIsComment) ? " (comment)" : "");
#ifdef GIF_LOAD_STATS
									if (bIsComment &&
										GIF_LOAD_STATS>=2 &&
										uNSkipped<sizeof(sDecomp.abyBuf) &&
										uNSkipped<(16*GIF_LOAD_STATS))
									{	gif_private_hexdump_data(sDecomp.abyBuf,uNSkipped-3);
									}
#endif
								}
								sDecomp.uNWarnings++;
							}
							sDecomp.abyBuf[0]=0;
						}
						break;
				}
				break;
			default:
				if (!sDecomp.bQuiet) fprintf(stderr,"Warning: Value 0x%02X un-expected at 0x%X\n",bLen,sDecomp.uReadBytes);
				sDecomp.uNWarnings++;
				break;
		}
	}

	if (psGifInfo->psImages[psGifInfo->uNImages-1].pLines==NULL)
	{	// Trailing control block
		if (!bQuiet) fprintf(stderr,"Warning: Trailing control block without image (deleted)\n");
		gif_free_image(&psGifInfo->psImages[psGifInfo->uNImages-1]);
		psGifInfo->uNImages--;
	}

#if (defined(GIF_LOAD_DEBUG) || defined(GIF_LOAD_STATS))
	printf("Read %u images in %u bytes. Screen is %ux%u %u bpp\n",
		psGifInfo->uNImages,sDecomp.uReadBytes,
		psGifInfo->screen.width,psGifInfo->screen.height,
		psGifInfo->screen.bitsPerPixelMinus1+1);
#endif

	gif_private_palette_signature_check(psGifInfo);
	{	unsigned uImage;
		for (uImage=0 ; uImage<psGifInfo->uNImages ; uImage++)
			if (psGifInfo->psImages[uImage].bHaveControl &&
				psGifInfo->psImages[uImage].control.disposal==3)
					break;
		psGifInfo->sOptions.bNoMode3=SET_BOOL(uImage>=psGifInfo->uNImages);
	}
	psGifInfo->uNWarnings=sDecomp.uNWarnings;
	return false;
}

// Find control block for this image
// Returns NULL if none exist
static gif_ext_graphics_control_label_t *gif_private_control_block(gif_file_t const *psGifInfo,unsigned uImage)
{	unsigned uControlBlock;
	for (uControlBlock=uImage ; uControlBlock>0 ; uControlBlock--)
		if (psGifInfo->psImages[uControlBlock].bHaveControl) break;
	return (psGifInfo->psImages[uControlBlock].bHaveControl) ? &psGifInfo->psImages[uControlBlock].control : NULL;
}

/************************** Public functions *******************************/

// Avoid problems with p=realloc(p,s) failing, as it doesn't allow p (or structures within p) to be freed
bool gif_realloc(void **ppvArray,size_t sSize)
{	void *pNew=realloc(*ppvArray,sSize);
	if (!pNew)
	{	//OUT_OF_MEMORY
#ifdef MEMORY_DEBUG
		memdebug_dump_all();
#endif
		return true;
	}
	*ppvArray=pNew;
	return false;
}

// Return true if image uses features only availible in GIF89a
bool gif_is_gif89(gif_file_t const *psGifInfo)
{	GIF_ASSERT(psGifInfo->uNImages>0);
	// More than one image, or special features in first image, then must be gif89
	if (psGifInfo->uNImages>1 ||
		psGifInfo->psImages[0].bHaveControl ||
		psGifInfo->psImages[0].bHaveNetscapeLoop) return true;
	return false;
}

// Create space for an extra image header, returns pointer to new entry, or NULL on failure
gif_image_t *gif_add_image_space(gif_file_t *psFile)
{	if (psFile->uNImages==0)
	{	psFile->psImages=malloc(sizeof(*psFile->psImages));
		if (psFile->psImages==NULL)
		{	gif_free(psFile);
			fprintf(stderr,"Error: Out of memory for image 0\n");
			return NULL;
		}
	} else
	if (gif_realloc((void **)&psFile->psImages,(psFile->uNImages+1)*sizeof(*psFile->psImages)))
	{	gif_free(psFile);
		fprintf(stderr,"Error: Out of memory for image %u\n",psFile->uNImages+1);
		return NULL;
	}
	memset(&psFile->psImages[psFile->uNImages],0,sizeof(psFile->psImages[psFile->uNImages]));
	psFile->psImages[psFile->uNImages].image.width=psFile->screen.width;
	psFile->psImages[psFile->uNImages].image.height=psFile->screen.height;
	psFile->uNImages++;
	return &psFile->psImages[psFile->uNImages-1];
}

// Allocate space for the actual image, based on information in header
bool gif_allocate_image_memory(gif_image_t *psImage)
{	unsigned uLine;
	BYTE *pData;

	psImage->uNPixels=psImage->image.width*psImage->image.height;
	if (psImage->image.width==0 || psImage->image.height==0)
	{	// Force image empty, incase not both zero
		psImage->image.width=psImage->image.height=0;
		psImage->pLines=malloc(sizeof(BYTE *));
		psImage->pLines[0]=NULL;
#ifdef GIF_LOAD_DEBUG
		fprintf(stderr,"Warning: Allocating space for empty image\n");
#endif
	} else
	{	GIF_ASSERT(psImage->image.width>0 && psImage->image.height>0);
		psImage->pLines=malloc(psImage->image.height*sizeof(BYTE *)+psImage->uNPixels);
	}

	if (!psImage->pLines)
	{	fprintf(stderr,"Error: Out of memory for %ux%u image\n",psImage->image.width,psImage->image.height);
		return true;
	}
	pData=((BYTE *)psImage->pLines)+psImage->image.height*sizeof(BYTE *);
	if (psImage->image.bInterlaced)
	{	// Rearrange line pointers to be in output order
		for (uLine=0 ; uLine<psImage->image.height ; uLine++)
		{	unsigned const uActualLine=gif_private_lookup_interlaced_line(&psImage->image,uLine);
			// Line uLine in the file is uActualLine on screen
			psImage->pLines[uActualLine]=pData+uLine*psImage->image.width;
		}
	} else
	{	for (uLine=0 ; uLine<psImage->image.height ; uLine++)
			psImage->pLines[uLine]=pData+uLine*psImage->image.width;
	}
	return false;
}

// Read data from a file, returns true on error
bool gif_read_data_from_file(BYTE *pDest,unsigned uNBytes,void *pDataHandle)
{	if (fread(pDest,uNBytes,1,(FILE *)pDataHandle)==1) return false;
	fprintf(stderr,"Error: Failed to read %u bytes from file\n",uNBytes);
	return true;
}

bool gif_read_images(gif_file_t *psGifInfo,
					bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle),
					void	*pDataHandle)
{	return gif_private_read_images(psGifInfo,false,read_data,pDataHandle);
}

bool gif_read_images_nowarns(gif_file_t *psGifInfo,
					bool	(*read_data)(BYTE *pDest,unsigned uNBytes,void *pDataHandle),
					void	*pDataHandle)
{	return gif_private_read_images(psGifInfo,false,read_data,pDataHandle);
}

// Read a gif file, from file with supplied name
bool gif_read_file(gif_file_t *pDest,const char *psFileName)
{	FILE *in=fopen(psFileName,"rb");
	if (in==NULL)
	{	fprintf(stderr,"Failed to open '%s'\n",psFileName);
		return true;
	}
	GIF_ASSERT(pDest);
	if (gif_read_images(pDest,gif_read_data_from_file,in))
	{	gif_free(pDest);
		fclose(in);
		return true;
	}
	fclose(in);
	return false;
}

// Delete data associated with a given image, mark data as freed
void gif_free_image(gif_image_t *psImage)
{	if (psImage->pLines)
	{	free(psImage->pLines);
		psImage->pLines=NULL;
		psImage->uNPixels=0;
	}
	if (psImage->psColourTable)
	{	free(psImage->psColourTable);
		psImage->psColourTable=NULL;
		psImage->uColourTableBytes=0;
	}
}

// Free the various structures within a gif_file_t
void gif_free(gif_file_t *psFile)
{	if (psFile->psImages)
	{	unsigned uImage;
		for (uImage=0 ; uImage<psFile->uNImages ; uImage++)
		{	gif_free_image(&psFile->psImages[uImage]);
		}
		free(psFile->psImages);
		psFile->psImages=NULL;
	}

	if (psFile->psColourTable)
	{	free(psFile->psColourTable);
		psFile->psColourTable=NULL;
		psFile->uColourTableBytes=0;
	}
}

#endif // NO_FUNCTIONS ]

#ifdef __cplusplus
}
#endif

#endif // DONE_GIF_LOAD_H ]
